﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sts/STS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace STS {
namespace Model {

/**
 * <p>You can pass custom key-value pair attributes when you assume a role or
 * federate a user. These are called session tags. You can then use the session
 * tags to control access to resources. For more information, see <a
 * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html">Tagging
 * Amazon Web Services STS Sessions</a> in the <i>IAM User Guide</i>.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sts-2011-06-15/Tag">AWS API
 * Reference</a></p>
 */
class Tag {
 public:
  AWS_STS_API Tag() = default;
  AWS_STS_API Tag(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_STS_API Tag& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_STS_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_STS_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The key for a session tag.</p> <p>You can pass up to 50 session tags. The
   * plain text session tag keys can’t exceed 128 characters. For these and
   * additional limits, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-limits.html#reference_iam-limits-entity-length">IAM
   * and STS Character Limits</a> in the <i>IAM User Guide</i>.</p>
   */
  inline const Aws::String& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::String>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::String>
  Tag& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value for a session tag.</p> <p>You can pass up to 50 session tags. The
   * plain text session tag values can’t exceed 256 characters. For these and
   * additional limits, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-limits.html#reference_iam-limits-entity-length">IAM
   * and STS Character Limits</a> in the <i>IAM User Guide</i>.</p>
   */
  inline const Aws::String& GetValue() const { return m_value; }
  inline bool ValueHasBeenSet() const { return m_valueHasBeenSet; }
  template <typename ValueT = Aws::String>
  void SetValue(ValueT&& value) {
    m_valueHasBeenSet = true;
    m_value = std::forward<ValueT>(value);
  }
  template <typename ValueT = Aws::String>
  Tag& WithValue(ValueT&& value) {
    SetValue(std::forward<ValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_key;
  bool m_keyHasBeenSet = false;

  Aws::String m_value;
  bool m_valueHasBeenSet = false;
};

}  // namespace Model
}  // namespace STS
}  // namespace Aws
