"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerGrammarChecker = void 0;
const tslib_1 = require("tslib");
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
// This file contains a lot of single setup logic for registering a language globally
const monaco_1 = require("../monaco");
const worker_proxy_service_1 = require("./worker_proxy_service");
const lexer_rules_1 = require("./lexer_rules");
const constants_1 = require("./constants");
const worker_store_1 = require("../worker_store");
// @ts-ignore
const xjson_editor_worker_js_1 = tslib_1.__importDefault(require("!!raw-loader!../../target/public/xjson.editor.worker.js"));
const wps = new worker_proxy_service_1.WorkerProxyService();
// Register rules against shared monaco instance.
lexer_rules_1.registerLexerRules(monaco_1.monaco);
// register xjson worker to the worker map.
worker_store_1.registerWorker(constants_1.ID, xjson_editor_worker_js_1.default);
monaco_1.monaco.languages.onLanguage(constants_1.ID, async () => {
    return wps.setup();
});
const OWNER = 'XJSON_GRAMMAR_CHECKER';
exports.registerGrammarChecker = () => {
    const allDisposables = [];
    const updateAnnotations = async (model) => {
        if (model.isDisposed()) {
            return;
        }
        const parseResult = await wps.getAnnos(model.uri);
        if (!parseResult) {
            return;
        }
        const { annotations } = parseResult;
        monaco_1.monaco.editor.setModelMarkers(model, OWNER, annotations.map(({ at, text, type }) => {
            const { column, lineNumber } = model.getPositionAt(at);
            return {
                startLineNumber: lineNumber,
                startColumn: column,
                endLineNumber: lineNumber,
                endColumn: column,
                message: text,
                severity: type === 'error' ? monaco_1.monaco.MarkerSeverity.Error : monaco_1.monaco.MarkerSeverity.Warning,
            };
        }));
    };
    const onModelAdd = (model) => {
        if (model.getModeId() === constants_1.ID) {
            allDisposables.push(model.onDidChangeContent(async () => {
                updateAnnotations(model);
            }));
            updateAnnotations(model);
        }
    };
    allDisposables.push(monaco_1.monaco.editor.onDidCreateModel(onModelAdd));
    return () => {
        wps.stop();
        allDisposables.forEach((d) => d.dispose());
    };
};
exports.registerGrammarChecker();
//# sourceMappingURL=language.js.map